/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.health;

import java.io.IOException;
import java.time.Clock;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.component.LifecycleListener;
import org.elasticsearch.common.logging.ESLogMessage;
import org.elasticsearch.common.scheduler.SchedulerEngine;
import org.elasticsearch.common.scheduler.TimeValueSchedule;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.health.HealthIndicatorResult;
import org.elasticsearch.health.HealthService;
import org.elasticsearch.health.HealthStatus;
import org.elasticsearch.health.node.selection.HealthNode;
import org.elasticsearch.telemetry.TelemetryProvider;
import org.elasticsearch.telemetry.metric.LongGaugeMetric;
import org.elasticsearch.telemetry.metric.MeterRegistry;

public class HealthPeriodicLogger
extends AbstractLifecycleComponent
implements ClusterStateListener,
SchedulerEngine.Listener {
    public static final String HEALTH_FIELD_PREFIX = "elasticsearch.health";
    public static final String MESSAGE_FIELD = "message";
    public static final Setting<TimeValue> POLL_INTERVAL_SETTING = Setting.timeSetting("health.periodic_logger.poll_interval", TimeValue.timeValueSeconds((long)60L), TimeValue.timeValueSeconds((long)15L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Boolean> ENABLED_SETTING = Setting.boolSetting("health.periodic_logger.enabled", false, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<List<OutputMode>> OUTPUT_MODE_SETTING = Setting.listSetting("health.periodic_logger.output_mode", List.of(OutputMode.LOGS.toString(), OutputMode.METRICS.toString()), OutputMode::parseOutputMode, Setting.Property.Dynamic, Setting.Property.NodeScope);
    protected static final String HEALTH_PERIODIC_LOGGER_JOB_NAME = "health_periodic_logger";
    private final Settings settings;
    private final ClusterService clusterService;
    private final Client client;
    private final HealthService healthService;
    private final Clock clock;
    private volatile boolean isHealthNode = false;
    private final Semaphore currentlyRunning = new Semaphore(1, true);
    private final SetOnce<SchedulerEngine> scheduler = new SetOnce();
    private volatile TimeValue pollInterval;
    private volatile boolean enabled;
    private volatile Set<OutputMode> outputModes;
    private static final Logger logger = LogManager.getLogger(HealthPeriodicLogger.class);
    private final MeterRegistry meterRegistry;
    private final Map<String, LongGaugeMetric> redMetrics = new HashMap<String, LongGaugeMetric>();
    private final BiConsumer<LongGaugeMetric, Long> metricWriter;
    private final Consumer<ESLogMessage> logWriter;
    final ActionListener<List<HealthIndicatorResult>> resultsListener = new ActionListener<List<HealthIndicatorResult>>(){

        @Override
        public void onResponse(List<HealthIndicatorResult> healthIndicatorResults) {
            try {
                Map<String, Object> resultsMap;
                if (HealthPeriodicLogger.this.logsEnabled() && !(resultsMap = HealthPeriodicLogger.convertToLoggedFields(healthIndicatorResults)).isEmpty()) {
                    ESLogMessage msg = new ESLogMessage().withFields(resultsMap);
                    HealthPeriodicLogger.this.logWriter.accept(msg);
                }
                if (HealthPeriodicLogger.this.metricsEnabled()) {
                    HealthPeriodicLogger.this.writeMetrics(healthIndicatorResults);
                }
            }
            catch (Exception e) {
                logger.warn("Health Periodic Logger error:{}", (Object)e.toString());
            }
        }

        @Override
        public void onFailure(Exception e) {
            logger.warn("Health Periodic Logger error:{}", (Object)e.toString());
        }
    };

    public static HealthPeriodicLogger create(Settings settings, ClusterService clusterService, Client client, HealthService healthService, TelemetryProvider telemetryProvider) {
        return HealthPeriodicLogger.create(settings, clusterService, client, healthService, telemetryProvider, null, null);
    }

    static HealthPeriodicLogger create(Settings settings, ClusterService clusterService, Client client, HealthService healthService, TelemetryProvider telemetryProvider, BiConsumer<LongGaugeMetric, Long> metricWriter, Consumer<ESLogMessage> logWriter) {
        HealthPeriodicLogger healthLogger = new HealthPeriodicLogger(settings, clusterService, client, healthService, telemetryProvider.getMeterRegistry(), metricWriter, logWriter);
        healthLogger.registerListeners();
        return healthLogger;
    }

    private HealthPeriodicLogger(Settings settings, ClusterService clusterService, Client client, HealthService healthService, MeterRegistry meterRegistry, BiConsumer<LongGaugeMetric, Long> metricWriter, Consumer<ESLogMessage> logWriter) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.client = client;
        this.healthService = healthService;
        this.clock = Clock.systemUTC();
        this.pollInterval = POLL_INTERVAL_SETTING.get(settings);
        this.enabled = ENABLED_SETTING.get(settings);
        this.outputModes = EnumSet.copyOf(OUTPUT_MODE_SETTING.get(settings));
        this.meterRegistry = meterRegistry;
        BiConsumer<LongGaugeMetric, Long> biConsumer = this.metricWriter = metricWriter == null ? LongGaugeMetric::set : metricWriter;
        this.logWriter = logWriter == null ? arg_0 -> ((Logger)logger).info(arg_0) : logWriter;
        this.redMetrics.put("overall", LongGaugeMetric.create(this.meterRegistry, "es.health.overall.red.status", "Overall: Red", "{cluster}"));
    }

    private void registerListeners() {
        if (this.enabled) {
            this.clusterService.addListener(this);
        }
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(ENABLED_SETTING, this::enable);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(POLL_INTERVAL_SETTING, this::updatePollInterval);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(OUTPUT_MODE_SETTING, this::updateOutputModes);
        this.addLifecycleListener(new LifecycleListener(){

            @Override
            public void afterStart() {
                HealthPeriodicLogger.this.maybeScheduleJob();
            }

            @Override
            public void afterStop() {
                HealthPeriodicLogger.this.maybeCancelJob();
            }
        });
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (event.state().blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            return;
        }
        DiscoveryNode healthNode = HealthNode.findHealthNode(event.state());
        if (healthNode == null) {
            this.isHealthNode = false;
            this.maybeCancelJob();
            return;
        }
        boolean isCurrentlyHealthNode = healthNode.getId().equals(this.clusterService.localNode().getId());
        if (this.isHealthNode != isCurrentlyHealthNode) {
            this.isHealthNode = isCurrentlyHealthNode;
            if (this.isHealthNode) {
                this.maybeScheduleJob();
            } else {
                this.maybeCancelJob();
            }
        }
    }

    @Override
    protected void doStart() {
        logger.debug("Periodic health logger is starting.");
    }

    @Override
    protected void doStop() {
        this.clusterService.removeListener(this);
        logger.debug("Periodic health logger is stopping.");
    }

    @Override
    protected void doClose() throws IOException {
        logger.debug("Periodic health logger is closing.");
        try {
            if (this.currentlyRunning.tryAcquire(2L, TimeUnit.SECONDS)) {
                logger.debug("Periodic health logger's last run has successfully finished.");
            }
        }
        catch (InterruptedException e) {
            logger.warn("Error while waiting for the last run of the periodic health logger to finish.", (Throwable)e);
        }
        finally {
            SchedulerEngine engine = (SchedulerEngine)this.scheduler.get();
            if (engine != null) {
                engine.stop();
            }
        }
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        if (event.getJobName().equals(HEALTH_PERIODIC_LOGGER_JOB_NAME) && this.enabled) {
            this.tryToLogHealth();
        }
    }

    boolean tryToLogHealth() {
        try {
            if (this.currentlyRunning.tryAcquire(0L, TimeUnit.SECONDS)) {
                RunOnce release = new RunOnce(this.currentlyRunning::release);
                try {
                    ActionListener<List<HealthIndicatorResult>> listenerWithRelease = ActionListener.runAfter(this.resultsListener, release);
                    this.healthService.getHealth(this.client, null, false, 0, listenerWithRelease);
                }
                catch (Exception e) {
                    release.run();
                    logger.warn(() -> "The health periodic logger encountered an error.", (Throwable)e);
                }
                return true;
            }
            logger.debug("Skipping this run because it's already in progress.");
        }
        catch (InterruptedException e) {
            logger.debug("Periodic health logger run was interrupted.", (Throwable)e);
        }
        return false;
    }

    SchedulerEngine getScheduler() {
        return (SchedulerEngine)this.scheduler.get();
    }

    static Map<String, Object> convertToLoggedFields(List<HealthIndicatorResult> indicatorResults) {
        if (indicatorResults == null || indicatorResults.isEmpty()) {
            return Map.of();
        }
        HashMap<String, Object> result = new HashMap<String, Object>();
        HealthStatus status = HealthPeriodicLogger.calculateOverallStatus(indicatorResults);
        result.put(String.format(Locale.ROOT, "%s.overall.status", HEALTH_FIELD_PREFIX), status.xContentValue());
        indicatorResults.forEach(indicatorResult -> result.put(String.format(Locale.ROOT, "%s.%s.status", HEALTH_FIELD_PREFIX, indicatorResult.name()), indicatorResult.status().xContentValue()));
        List<String> nonGreen = indicatorResults.stream().filter(p -> p.status() != HealthStatus.GREEN).map(HealthIndicatorResult::name).sorted().toList();
        if (nonGreen.isEmpty()) {
            result.put(MESSAGE_FIELD, String.format(Locale.ROOT, "health=%s", status.xContentValue()));
        } else {
            result.put(MESSAGE_FIELD, String.format(Locale.ROOT, "health=%s [%s]", status.xContentValue(), String.join((CharSequence)",", nonGreen)));
        }
        return result;
    }

    static HealthStatus calculateOverallStatus(List<HealthIndicatorResult> indicatorResults) {
        return HealthStatus.merge(indicatorResults.stream().map(HealthIndicatorResult::status));
    }

    void writeMetrics(List<HealthIndicatorResult> healthIndicatorResults) {
        if (healthIndicatorResults != null) {
            for (HealthIndicatorResult result : healthIndicatorResults) {
                String metricName = result.name();
                LongGaugeMetric metric = this.redMetrics.get(metricName);
                if (metric == null) {
                    metric = LongGaugeMetric.create(this.meterRegistry, String.format(Locale.ROOT, "es.health.%s.red.status", metricName), String.format(Locale.ROOT, "%s: Red", metricName), "{cluster}");
                    this.redMetrics.put(metricName, metric);
                }
                this.metricWriter.accept(metric, result.status() == HealthStatus.RED ? 1L : 0L);
            }
            this.metricWriter.accept(this.redMetrics.get("overall"), HealthPeriodicLogger.calculateOverallStatus(healthIndicatorResults) == HealthStatus.RED ? 1L : 0L);
        }
    }

    private void updateOutputModes(List<OutputMode> newMode) {
        this.outputModes = EnumSet.copyOf(newMode);
    }

    private boolean logsEnabled() {
        return this.outputModes.contains((Object)OutputMode.LOGS);
    }

    private boolean metricsEnabled() {
        return this.outputModes.contains((Object)OutputMode.METRICS);
    }

    private void maybeScheduleJob() {
        if (!this.isHealthNode) {
            return;
        }
        if (!this.enabled) {
            return;
        }
        if (!this.isStarted()) {
            logger.trace("Skipping scheduling a health periodic logger job due to the health logger lifecycle state being: [{}] ", (Object)this.lifecycleState());
            return;
        }
        if (this.scheduler.get() == null) {
            this.scheduler.set((Object)new SchedulerEngine(this.settings, this.clock));
            ((SchedulerEngine)this.scheduler.get()).register(this);
        }
        assert (this.scheduler.get() != null) : "scheduler should be available";
        SchedulerEngine.Job scheduledJob = new SchedulerEngine.Job(HEALTH_PERIODIC_LOGGER_JOB_NAME, new TimeValueSchedule(this.pollInterval));
        ((SchedulerEngine)this.scheduler.get()).add(scheduledJob);
    }

    private void maybeCancelJob() {
        if (this.scheduler.get() != null) {
            ((SchedulerEngine)this.scheduler.get()).remove(HEALTH_PERIODIC_LOGGER_JOB_NAME);
        }
    }

    private void enable(boolean enabled) {
        this.enabled = enabled;
        if (enabled & !this.isStoppedOrClosed()) {
            this.clusterService.addListener(this);
            this.maybeScheduleJob();
        } else {
            this.clusterService.removeListener(this);
            this.maybeCancelJob();
        }
    }

    private void updatePollInterval(TimeValue newInterval) {
        this.pollInterval = newInterval;
        if (!this.isStoppedOrClosed()) {
            this.maybeScheduleJob();
        }
    }

    private boolean isStarted() {
        return this.lifecycleState() == Lifecycle.State.STARTED;
    }

    private boolean isStoppedOrClosed() {
        return this.lifecycleState() == Lifecycle.State.STOPPED || this.lifecycleState() == Lifecycle.State.CLOSED;
    }

    TimeValue getPollInterval() {
        return this.pollInterval;
    }

    boolean isHealthNode() {
        return this.isHealthNode;
    }

    boolean enabled() {
        return this.enabled;
    }

    boolean currentlyRunning() {
        return this.currentlyRunning.availablePermits() == 0;
    }

    boolean waitingToFinishCurrentRun() {
        return this.currentlyRunning.hasQueuedThreads();
    }

    public static enum OutputMode {
        LOGS("logs"),
        METRICS("metrics");

        private final String mode;

        private OutputMode(String mode) {
            this.mode = mode;
        }

        public static OutputMode fromString(String mode) {
            return OutputMode.valueOf(mode.toUpperCase(Locale.ROOT));
        }

        public String toString() {
            return this.mode.toLowerCase(Locale.ROOT);
        }

        static OutputMode parseOutputMode(String value) {
            try {
                return OutputMode.fromString(value);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Illegal OutputMode:" + value);
            }
        }
    }
}

