/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.store;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateObserver;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestHandler;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public final class IndicesStore
implements ClusterStateListener,
Closeable {
    private static final Logger logger = LogManager.getLogger(IndicesStore.class);
    public static final Setting<TimeValue> INDICES_STORE_DELETE_SHARD_TIMEOUT = Setting.positiveTimeSetting("indices.store.delete.shard.timeout", new TimeValue(30L, TimeUnit.SECONDS), Setting.Property.NodeScope);
    public static final String ACTION_SHARD_EXISTS = "internal:index/shard/exists";
    private static final EnumSet<IndexShardState> ACTIVE_STATES = EnumSet.of(IndexShardState.STARTED);
    private final Settings settings;
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final ThreadPool threadPool;
    private final Set<ShardId> folderNotFoundCache = new HashSet<ShardId>();
    private final TimeValue deleteShardTimeout;

    @Inject
    public IndicesStore(Settings settings, IndicesService indicesService, ClusterService clusterService, TransportService transportService, ThreadPool threadPool) {
        this.settings = settings;
        this.indicesService = indicesService;
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.threadPool = threadPool;
        transportService.registerRequestHandler(ACTION_SHARD_EXISTS, EsExecutors.DIRECT_EXECUTOR_SERVICE, ShardActiveRequest::new, new ShardActiveRequestHandler());
        this.deleteShardTimeout = INDICES_STORE_DELETE_SHARD_TIMEOUT.get(settings);
        if (DiscoveryNode.canContainData(settings)) {
            clusterService.addListener(this);
        }
    }

    @Override
    public void close() {
        if (DiscoveryNode.canContainData(this.settings)) {
            this.clusterService.removeListener(this);
        }
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.routingTableChanged()) {
            return;
        }
        if (event.state().blocks().disableStatePersistence()) {
            return;
        }
        RoutingTable routingTable = event.state().routingTable();
        this.folderNotFoundCache.removeIf(shardId -> !routingTable.hasIndex(shardId.getIndex()));
        String localNodeId = event.state().nodes().getLocalNodeId();
        RoutingNode localRoutingNode = event.state().getRoutingNodes().node(localNodeId);
        if (localRoutingNode != null) {
            for (ShardRouting routing : localRoutingNode) {
                this.folderNotFoundCache.remove(routing.shardId());
            }
        }
        for (IndexRoutingTable indexRoutingTable : routingTable) {
            block7: for (int i = 0; i < indexRoutingTable.size(); ++i) {
                IndexSettings indexSettings;
                IndexShardRoutingTable indexShardRoutingTable = indexRoutingTable.shard(i);
                ShardId shardId2 = indexShardRoutingTable.shardId();
                if (this.folderNotFoundCache.contains(shardId2) || !IndicesStore.shardCanBeDeleted(localNodeId, indexShardRoutingTable)) continue;
                IndexService indexService = this.indicesService.indexService(indexRoutingTable.getIndex());
                if (indexService == null) {
                    IndexMetadata indexMetadata = event.state().getMetadata().getIndexSafe(indexRoutingTable.getIndex());
                    indexSettings = new IndexSettings(indexMetadata, this.settings);
                } else {
                    indexSettings = indexService.getIndexSettings();
                }
                IndicesService.ShardDeletionCheckResult shardDeletionCheckResult = this.indicesService.canDeleteShardContent(shardId2, indexSettings);
                switch (shardDeletionCheckResult) {
                    case FOLDER_FOUND_CAN_DELETE: {
                        this.deleteShardIfExistElseWhere(event.state(), indexShardRoutingTable);
                        continue block7;
                    }
                    case NO_FOLDER_FOUND: {
                        this.folderNotFoundCache.add(shardId2);
                        continue block7;
                    }
                    case STILL_ALLOCATED: {
                        continue block7;
                    }
                    default: {
                        assert (false) : "unknown shard deletion check result: " + shardDeletionCheckResult;
                        continue block7;
                    }
                }
            }
        }
    }

    static boolean shardCanBeDeleted(String localNodeId, IndexShardRoutingTable indexShardRoutingTable) {
        assert (indexShardRoutingTable.size() > 0);
        if (indexShardRoutingTable.size() == 0) {
            return false;
        }
        for (int copy = 0; copy < indexShardRoutingTable.size(); ++copy) {
            ShardRouting shardRouting = indexShardRoutingTable.shard(copy);
            if (!shardRouting.started()) {
                return false;
            }
            if (!localNodeId.equals(shardRouting.currentNodeId())) continue;
            return false;
        }
        return true;
    }

    private void deleteShardIfExistElseWhere(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        ArrayList<Tuple> requests = new ArrayList<Tuple>(indexShardRoutingTable.size());
        String indexUUID = indexShardRoutingTable.shardId().getIndex().getUUID();
        ClusterName clusterName = state.getClusterName();
        for (int copy = 0; copy < indexShardRoutingTable.size(); ++copy) {
            ShardRouting shardRouting = indexShardRoutingTable.shard(copy);
            assert (shardRouting.started()) : "expected started shard but was " + shardRouting;
            DiscoveryNode currentNode = state.nodes().get(shardRouting.currentNodeId());
            requests.add(new Tuple((Object)currentNode, (Object)new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId(), this.deleteShardTimeout)));
        }
        ShardActiveResponseHandler responseHandler = new ShardActiveResponseHandler(indexShardRoutingTable.shardId(), state.getVersion(), requests.size());
        for (Tuple request : requests) {
            logger.trace("{} sending shard active check to {}", (Object)((ShardActiveRequest)request.v2()).shardId, request.v1());
            this.transportService.sendRequest((DiscoveryNode)request.v1(), ACTION_SHARD_EXISTS, (TransportRequest)request.v2(), responseHandler);
        }
    }

    private class ShardActiveRequestHandler
    implements TransportRequestHandler<ShardActiveRequest> {
        private ShardActiveRequestHandler() {
        }

        @Override
        public void messageReceived(final ShardActiveRequest request, final TransportChannel channel, Task task) {
            IndexShard indexShard = this.getShard(request);
            if (indexShard == null) {
                channel.sendResponse(new ShardActiveResponse(false, IndicesStore.this.clusterService.localNode()));
            } else {
                ClusterStateObserver observer = new ClusterStateObserver(IndicesStore.this.clusterService, request.timeout, logger, IndicesStore.this.threadPool.getThreadContext());
                boolean shardActive = ShardActiveRequestHandler.shardActive(indexShard);
                if (shardActive) {
                    channel.sendResponse(new ShardActiveResponse(true, IndicesStore.this.clusterService.localNode()));
                } else {
                    observer.waitForNextChange(new ClusterStateObserver.Listener(){

                        @Override
                        public void onNewClusterState(ClusterState state) {
                            this.sendResult(ShardActiveRequestHandler.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        @Override
                        public void onClusterServiceClose() {
                            this.sendResult(false);
                        }

                        @Override
                        public void onTimeout(TimeValue timeout) {
                            this.sendResult(ShardActiveRequestHandler.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        public void sendResult(boolean shardActive) {
                            try {
                                channel.sendResponse(new ShardActiveResponse(shardActive, IndicesStore.this.clusterService.localNode()));
                            }
                            catch (EsRejectedExecutionException e) {
                                logger.error(() -> Strings.format((String)"failed send response for shard active while trying to delete shard %s - shard will probably not be removed", (Object[])new Object[]{request2.shardId}), (Throwable)e);
                            }
                        }
                    }, newState -> {
                        IndexShard currentShard = this.getShard(request);
                        return currentShard == null || ShardActiveRequestHandler.shardActive(currentShard);
                    });
                }
            }
        }

        private static boolean shardActive(IndexShard indexShard) {
            if (indexShard != null) {
                return ACTIVE_STATES.contains((Object)indexShard.state());
            }
            return false;
        }

        private IndexShard getShard(ShardActiveRequest request) {
            ClusterName thisClusterName = IndicesStore.this.clusterService.getClusterName();
            if (!thisClusterName.equals(request.clusterName)) {
                logger.trace("shard exists request meant for cluster[{}], but this is cluster[{}], ignoring request", (Object)request.clusterName, (Object)thisClusterName);
                return null;
            }
            ShardId shardId = request.shardId;
            IndexService indexService = IndicesStore.this.indicesService.indexService(shardId.getIndex());
            if (indexService != null && indexService.indexUUID().equals(request.indexUUID)) {
                return indexService.getShardOrNull(shardId.id());
            }
            return null;
        }
    }

    private static class ShardActiveRequest
    extends TransportRequest {
        private final TimeValue timeout;
        private final ClusterName clusterName;
        private final String indexUUID;
        private final ShardId shardId;

        ShardActiveRequest(StreamInput in) throws IOException {
            super(in);
            this.clusterName = new ClusterName(in);
            this.indexUUID = in.readString();
            this.shardId = new ShardId(in);
            this.timeout = new TimeValue(in.readLong(), TimeUnit.MILLISECONDS);
        }

        ShardActiveRequest(ClusterName clusterName, String indexUUID, ShardId shardId, TimeValue timeout) {
            this.shardId = shardId;
            this.indexUUID = indexUUID;
            this.clusterName = clusterName;
            this.timeout = timeout;
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.clusterName.writeTo(out);
            out.writeString(this.indexUUID);
            this.shardId.writeTo(out);
            out.writeLong(this.timeout.millis());
        }
    }

    private class ShardActiveResponseHandler
    implements TransportResponseHandler<ShardActiveResponse> {
        private final ShardId shardId;
        private final int expectedActiveCopies;
        private final long clusterStateVersion;
        private final AtomicInteger awaitingResponses;
        private final AtomicInteger activeCopies;

        ShardActiveResponseHandler(ShardId shardId, long clusterStateVersion, int expectedActiveCopies) {
            this.shardId = shardId;
            this.expectedActiveCopies = expectedActiveCopies;
            this.clusterStateVersion = clusterStateVersion;
            this.awaitingResponses = new AtomicInteger(expectedActiveCopies);
            this.activeCopies = new AtomicInteger();
        }

        @Override
        public ShardActiveResponse read(StreamInput in) throws IOException {
            return new ShardActiveResponse(in);
        }

        @Override
        public Executor executor() {
            return TransportResponseHandler.TRANSPORT_WORKER;
        }

        @Override
        public void handleResponse(ShardActiveResponse response) {
            logger.trace("{} is {}active on node {}", (Object)this.shardId, (Object)(response.shardActive ? "" : "not "), (Object)response.node);
            if (response.shardActive) {
                this.activeCopies.incrementAndGet();
            }
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public void handleException(TransportException exp) {
            logger.debug(() -> Strings.format((String)"shards active request failed for %s", (Object[])new Object[]{this.shardId}), (Throwable)exp);
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        private void allNodesResponded() {
            if (this.activeCopies.get() != this.expectedActiveCopies) {
                logger.trace("not deleting shard {}, expected {} active copies, but only {} found active copies", (Object)this.shardId, (Object)this.expectedActiveCopies, (Object)this.activeCopies.get());
                return;
            }
            ClusterState latestClusterState = IndicesStore.this.clusterService.state();
            if (this.clusterStateVersion != latestClusterState.getVersion()) {
                logger.trace("not deleting shard {}, the latest cluster state version[{}] is not equal to cluster state before shard active api call [{}]", (Object)this.shardId, (Object)latestClusterState.getVersion(), (Object)this.clusterStateVersion);
                return;
            }
            IndicesStore.this.clusterService.getClusterApplierService().runOnApplierThread("indices_store ([" + this.shardId + "] active fully on other nodes)", Priority.HIGH, currentState -> {
                if (this.clusterStateVersion != currentState.getVersion()) {
                    logger.trace("not deleting shard {}, the update task state version[{}] is not equal to cluster state before shard active api call [{}]", (Object)this.shardId, (Object)currentState.getVersion(), (Object)this.clusterStateVersion);
                    return;
                }
                try {
                    IndicesStore.this.indicesService.deleteShardStore("no longer used", this.shardId, (ClusterState)currentState);
                }
                catch (Exception ex) {
                    logger.debug(() -> Strings.format((String)"%s failed to delete unallocated shard, ignoring", (Object[])new Object[]{this.shardId}), (Throwable)ex);
                }
            }, new ActionListener<Void>(){

                @Override
                public void onResponse(Void unused) {
                }

                @Override
                public void onFailure(Exception e) {
                    logger.error(() -> Strings.format((String)"%s unexpected error during deletion of unallocated shard", (Object[])new Object[]{ShardActiveResponseHandler.this.shardId}), (Throwable)e);
                }
            });
        }
    }

    private static class ShardActiveResponse
    extends TransportResponse {
        private final boolean shardActive;
        private final DiscoveryNode node;

        ShardActiveResponse(boolean shardActive, DiscoveryNode node) {
            this.shardActive = shardActive;
            this.node = node;
        }

        ShardActiveResponse(StreamInput in) throws IOException {
            this.shardActive = in.readBoolean();
            this.node = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeBoolean(this.shardActive);
            this.node.writeTo(out);
        }
    }
}

