/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.coordination;

import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.logging.log4j.util.MessageSupplier;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.coordination.CoordinationStateRejectedException;
import org.elasticsearch.cluster.coordination.Coordinator;
import org.elasticsearch.cluster.coordination.NodeHealthCheckFailureException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.discovery.zen.MasterFaultDetection;
import org.elasticsearch.monitor.NodeHealthService;
import org.elasticsearch.monitor.StatusInfo;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.NodeDisconnectedException;
import org.elasticsearch.transport.ReceiveTimeoutTransportException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportConnectionListener;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class LeaderChecker {
    private static final Logger logger = LogManager.getLogger(LeaderChecker.class);
    static final String LEADER_CHECK_ACTION_NAME = "internal:coordination/fault_detection/leader_check";
    public static final Setting<TimeValue> LEADER_CHECK_INTERVAL_SETTING = Setting.timeSetting("cluster.fault_detection.leader_check.interval", TimeValue.timeValueMillis((long)1000L), TimeValue.timeValueMillis((long)100L), Setting.Property.NodeScope);
    public static final Setting<TimeValue> LEADER_CHECK_TIMEOUT_SETTING = Setting.timeSetting("cluster.fault_detection.leader_check.timeout", TimeValue.timeValueMillis((long)10000L), TimeValue.timeValueMillis((long)1L), Setting.Property.NodeScope);
    public static final Setting<Integer> LEADER_CHECK_RETRY_COUNT_SETTING = Setting.intSetting("cluster.fault_detection.leader_check.retry_count", 3, 1, Setting.Property.NodeScope);
    private final Settings settings;
    private final TimeValue leaderCheckInterval;
    private final TimeValue leaderCheckTimeout;
    private final int leaderCheckRetryCount;
    private final TransportService transportService;
    private final LeaderFailureListener leaderFailureListener;
    private final NodeHealthService nodeHealthService;
    private final AtomicReference<CheckScheduler> currentChecker = new AtomicReference();
    private volatile DiscoveryNodes discoveryNodes;
    private static final String RESTARTING_DISCOVERY_TEXT = "restarting discovery; more details may be available in the master node logs";

    LeaderChecker(Settings settings, TransportService transportService, LeaderFailureListener leaderFailureListener, NodeHealthService nodeHealthService) {
        this.settings = settings;
        this.leaderCheckInterval = LEADER_CHECK_INTERVAL_SETTING.get(settings);
        this.leaderCheckTimeout = LEADER_CHECK_TIMEOUT_SETTING.get(settings);
        this.leaderCheckRetryCount = LEADER_CHECK_RETRY_COUNT_SETTING.get(settings);
        this.transportService = transportService;
        this.leaderFailureListener = leaderFailureListener;
        this.nodeHealthService = nodeHealthService;
        transportService.registerRequestHandler(LEADER_CHECK_ACTION_NAME, "same", false, false, LeaderCheckRequest::new, (request, channel, task) -> {
            this.handleLeaderCheck((LeaderCheckRequest)request);
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        });
        transportService.registerRequestHandler("internal:discovery/zen/fd/master_ping", "same", false, false, MasterFaultDetection.MasterPingRequest::new, (request, channel, task) -> {
            try {
                this.handleLeaderCheck(new LeaderCheckRequest(request.sourceNode));
            }
            catch (CoordinationStateRejectedException e) {
                throw new MasterFaultDetection.ThisIsNotTheMasterYouAreLookingForException(e.getMessage());
            }
            channel.sendResponse(new MasterFaultDetection.MasterPingResponseResponse());
        });
        transportService.addConnectionListener(new TransportConnectionListener(){

            @Override
            public void onNodeDisconnected(DiscoveryNode node, Transport.Connection connection) {
                LeaderChecker.this.handleDisconnectedNode(node);
            }
        });
    }

    public DiscoveryNode leader() {
        CheckScheduler checkScheduler = this.currentChecker.get();
        return checkScheduler == null ? null : checkScheduler.leader;
    }

    void updateLeader(@Nullable DiscoveryNode leader) {
        assert (!this.transportService.getLocalNode().equals(leader));
        CheckScheduler checkScheduler = leader != null ? new CheckScheduler(leader) : null;
        CheckScheduler previousChecker = this.currentChecker.getAndSet(checkScheduler);
        if (previousChecker != null) {
            previousChecker.close();
        }
        if (checkScheduler != null) {
            checkScheduler.handleWakeUp();
        }
    }

    void setCurrentNodes(DiscoveryNodes discoveryNodes) {
        logger.trace("setCurrentNodes: {}", (Object)discoveryNodes);
        this.discoveryNodes = discoveryNodes;
    }

    boolean currentNodeIsMaster() {
        return this.discoveryNodes.isLocalNodeElectedMaster();
    }

    private void handleLeaderCheck(LeaderCheckRequest request) {
        DiscoveryNodes discoveryNodes = this.discoveryNodes;
        assert (discoveryNodes != null);
        StatusInfo statusInfo = this.nodeHealthService.getHealth();
        if (statusInfo.getStatus() == StatusInfo.Status.UNHEALTHY) {
            logger.debug("this node is unhealthy [{}], rejecting leader check: {}", (Object)statusInfo.getInfo(), (Object)request);
            throw new NodeHealthCheckFailureException(statusInfo.getInfo(), new Object[0]);
        }
        if (!discoveryNodes.isLocalNodeElectedMaster()) {
            logger.debug("rejecting leader check on non-master: {}", (Object)request);
            throw new CoordinationStateRejectedException("no longer the elected master", new Object[0]);
        }
        if (!discoveryNodes.nodeExists(request.getSender())) {
            logger.debug("rejecting leader check from removed node: {}", (Object)request);
            throw new CoordinationStateRejectedException("rejecting check since [" + request.getSender().descriptionWithoutAttributes() + "] has been removed from the cluster", new Object[0]);
        }
        logger.trace("handling {}", (Object)request);
    }

    private void handleDisconnectedNode(DiscoveryNode discoveryNode) {
        CheckScheduler checkScheduler = this.currentChecker.get();
        if (checkScheduler != null) {
            checkScheduler.handleDisconnectedNode(discoveryNode);
        } else {
            logger.trace("disconnect event ignored for {}, no check scheduler", (Object)discoveryNode);
        }
    }

    @FunctionalInterface
    static interface LeaderFailureListener {
        public void onLeaderFailure(MessageSupplier var1, Exception var2);
    }

    private class CheckScheduler
    implements Releasable {
        private final DiscoveryNode leader;
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private int rejectedCountSinceLastSuccess;
        private int timeoutCountSinceLastSuccess;

        CheckScheduler(DiscoveryNode leader) {
            this.leader = leader;
        }

        public void close() {
            if (!this.isClosed.compareAndSet(false, true)) {
                logger.trace("already closed, doing nothing");
            } else {
                logger.debug("closed");
            }
        }

        void handleWakeUp() {
            TransportRequest transportRequest;
            String actionName;
            if (this.isClosed.get()) {
                logger.trace("closed check scheduler woken up, doing nothing");
                return;
            }
            logger.trace("checking {} with [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_TIMEOUT_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckTimeout);
            if (Coordinator.isZen1Node(this.leader)) {
                actionName = "internal:discovery/zen/fd/master_ping";
                transportRequest = new MasterFaultDetection.MasterPingRequest(LeaderChecker.this.transportService.getLocalNode(), this.leader, ClusterName.CLUSTER_NAME_SETTING.get(LeaderChecker.this.settings));
            } else {
                actionName = LeaderChecker.LEADER_CHECK_ACTION_NAME;
                transportRequest = new LeaderCheckRequest(LeaderChecker.this.transportService.getLocalNode());
            }
            LeaderChecker.this.transportService.sendRequest(this.leader, actionName, transportRequest, TransportRequestOptions.of(LeaderChecker.this.leaderCheckTimeout, TransportRequestOptions.Type.PING), new TransportResponseHandler.Empty(){

                @Override
                public void handleResponse(TransportResponse.Empty response) {
                    if (CheckScheduler.this.isClosed.get()) {
                        logger.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    CheckScheduler.this.rejectedCountSinceLastSuccess = 0;
                    CheckScheduler.this.timeoutCountSinceLastSuccess = 0;
                    CheckScheduler.this.scheduleNextWakeUp();
                }

                @Override
                public void handleException(TransportException exp) {
                    if (CheckScheduler.this.isClosed.get()) {
                        logger.debug("closed check scheduler received a response, doing nothing");
                        return;
                    }
                    if (exp instanceof ConnectTransportException || exp.getCause() instanceof ConnectTransportException) {
                        logger.debug((Message)new ParameterizedMessage("leader [{}] disconnected during check", (Object)CheckScheduler.this.leader), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(() -> new ParameterizedMessage("master node [{}] disconnected, restarting discovery [{}]", (Object)CheckScheduler.this.leader.descriptionWithoutAttributes(), (Object)ExceptionsHelper.unwrapCause(exp).getMessage()), exp);
                        return;
                    }
                    if (exp.getCause() instanceof NodeHealthCheckFailureException) {
                        logger.debug((Message)new ParameterizedMessage("leader [{}] health check failed", (Object)CheckScheduler.this.leader), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(() -> new ParameterizedMessage("master node [{}] reported itself as unhealthy [{}], {}", new Object[]{CheckScheduler.this.leader.descriptionWithoutAttributes(), exp.getCause().getMessage(), LeaderChecker.RESTARTING_DISCOVERY_TEXT}), exp);
                        return;
                    }
                    if (exp instanceof ReceiveTimeoutTransportException) {
                        CheckScheduler.this.timeoutCountSinceLastSuccess += 1;
                    } else {
                        CheckScheduler.this.rejectedCountSinceLastSuccess += 1;
                    }
                    long failureCount = CheckScheduler.this.rejectedCountSinceLastSuccess + CheckScheduler.this.timeoutCountSinceLastSuccess;
                    if (failureCount >= (long)LeaderChecker.this.leaderCheckRetryCount) {
                        logger.debug((Message)new ParameterizedMessage("leader [{}] failed {} consecutive checks (rejected [{}], timed out [{}], limit [{}] is {})", new Object[]{CheckScheduler.this.leader, failureCount, CheckScheduler.this.rejectedCountSinceLastSuccess, CheckScheduler.this.timeoutCountSinceLastSuccess, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount}), (Throwable)exp);
                        CheckScheduler.this.leaderFailed(() -> new ParameterizedMessage("[{}] consecutive checks of the master node [{}] were unsuccessful ([{}] rejected, [{}] timed out), {} [last unsuccessful check: {}]", new Object[]{failureCount, CheckScheduler.this.leader.descriptionWithoutAttributes(), CheckScheduler.this.rejectedCountSinceLastSuccess, CheckScheduler.this.timeoutCountSinceLastSuccess, LeaderChecker.RESTARTING_DISCOVERY_TEXT, ExceptionsHelper.unwrapCause(exp).getMessage()}), exp);
                        return;
                    }
                    logger.debug((Message)new ParameterizedMessage("{} consecutive failures (limit [{}] is {}) with leader [{}]", new Object[]{failureCount, LEADER_CHECK_RETRY_COUNT_SETTING.getKey(), LeaderChecker.this.leaderCheckRetryCount, CheckScheduler.this.leader}), (Throwable)exp);
                    CheckScheduler.this.scheduleNextWakeUp();
                }
            });
        }

        void leaderFailed(final MessageSupplier messageSupplier, final Exception e) {
            if (this.isClosed.compareAndSet(false, true)) {
                LeaderChecker.this.transportService.getThreadPool().generic().execute(new Runnable(){

                    @Override
                    public void run() {
                        LeaderChecker.this.leaderFailureListener.onLeaderFailure(messageSupplier, e);
                    }

                    public String toString() {
                        return "notification of leader failure: " + e.getMessage();
                    }
                });
            } else {
                logger.trace("already closed, not failing leader");
            }
        }

        void handleDisconnectedNode(DiscoveryNode discoveryNode) {
            if (discoveryNode.equals(this.leader)) {
                logger.debug("leader [{}] disconnected", (Object)this.leader);
                this.leaderFailed(() -> new ParameterizedMessage("master node [{}] disconnected, restarting discovery", (Object)this.leader.descriptionWithoutAttributes()), new NodeDisconnectedException(discoveryNode, "disconnected"));
            }
        }

        private void scheduleNextWakeUp() {
            logger.trace("scheduling next check of {} for [{}] = {}", (Object)this.leader, (Object)LEADER_CHECK_INTERVAL_SETTING.getKey(), (Object)LeaderChecker.this.leaderCheckInterval);
            LeaderChecker.this.transportService.getThreadPool().schedule(new Runnable(){

                @Override
                public void run() {
                    CheckScheduler.this.handleWakeUp();
                }

                public String toString() {
                    return "scheduled check of leader " + CheckScheduler.this.leader;
                }
            }, LeaderChecker.this.leaderCheckInterval, "same");
        }
    }

    static class LeaderCheckRequest
    extends TransportRequest {
        private final DiscoveryNode sender;

        LeaderCheckRequest(DiscoveryNode sender) {
            this.sender = sender;
        }

        LeaderCheckRequest(StreamInput in) throws IOException {
            super(in);
            this.sender = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.sender.writeTo(out);
        }

        public DiscoveryNode getSender() {
            return this.sender;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LeaderCheckRequest that = (LeaderCheckRequest)o;
            return Objects.equals(this.sender, that.sender);
        }

        public int hashCode() {
            return Objects.hash(this.sender);
        }

        @Override
        public String toString() {
            return "LeaderCheckRequest{sender=" + this.sender + '}';
        }
    }
}

