/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.http;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.http.HttpChannel;
import org.elasticsearch.http.HttpRequest;
import org.elasticsearch.http.HttpStats;
import org.elasticsearch.http.HttpTransportSettings;
import org.elasticsearch.threadpool.ThreadPool;

public class HttpClientStatsTracker {
    private static final Logger logger = LogManager.getLogger();
    private static final long PRUNE_THROTTLE_INTERVAL = TimeUnit.SECONDS.toMillis(60L);
    private static final long MAX_CLIENT_STATS_AGE = TimeUnit.MINUTES.toMillis(5L);
    private final Map<Integer, HttpStats.ClientStats> httpChannelStats = new ConcurrentHashMap<Integer, HttpStats.ClientStats>();
    private final ThreadPool threadPool;
    private volatile long lastClientStatsPruneTime;
    private volatile boolean clientStatsEnabled;

    HttpClientStatsTracker(Settings settings, ClusterSettings clusterSettings, ThreadPool threadPool) {
        this.threadPool = threadPool;
        this.clientStatsEnabled = HttpTransportSettings.SETTING_HTTP_CLIENT_STATS_ENABLED.get(settings);
        clusterSettings.addSettingsUpdateConsumer(HttpTransportSettings.SETTING_HTTP_CLIENT_STATS_ENABLED, this::enableClientStats);
    }

    private void pruneClientStats(boolean throttled) {
        if (this.clientStatsEnabled && !throttled || this.threadPool.relativeTimeInMillis() - this.lastClientStatsPruneTime > PRUNE_THROTTLE_INTERVAL) {
            long nowMillis = this.threadPool.absoluteTimeInMillis();
            for (Map.Entry<Integer, HttpStats.ClientStats> statsEntry : this.httpChannelStats.entrySet()) {
                long closedTimeMillis = statsEntry.getValue().closedTimeMillis;
                if (closedTimeMillis <= 0L || nowMillis - closedTimeMillis <= MAX_CLIENT_STATS_AGE) continue;
                this.httpChannelStats.remove(statsEntry.getKey());
            }
            this.lastClientStatsPruneTime = this.threadPool.relativeTimeInMillis();
        }
    }

    private void enableClientStats(boolean enabled) {
        this.clientStatsEnabled = enabled;
        if (!enabled) {
            this.httpChannelStats.clear();
        }
    }

    HttpStats.ClientStats addClientStats(HttpChannel httpChannel) {
        if (this.clientStatsEnabled) {
            HttpStats.ClientStats clientStats;
            if (httpChannel != null) {
                clientStats = new HttpStats.ClientStats(this.threadPool.absoluteTimeInMillis());
                this.httpChannelStats.put(HttpClientStatsTracker.getChannelKey(httpChannel), clientStats);
                httpChannel.addCloseListener(ActionListener.wrap(() -> {
                    try {
                        HttpStats.ClientStats disconnectedClientStats = this.httpChannelStats.get(HttpClientStatsTracker.getChannelKey(httpChannel));
                        if (disconnectedClientStats != null) {
                            disconnectedClientStats.closedTimeMillis = this.threadPool.absoluteTimeInMillis();
                        }
                    }
                    catch (Exception e) {
                        assert (false) : e;
                        logger.warn("error removing HTTP channel listener", (Throwable)e);
                    }
                }));
            } else {
                clientStats = null;
            }
            this.pruneClientStats(true);
            return clientStats;
        }
        return null;
    }

    private static String getFirstValueForHeader(HttpRequest request, String header) {
        for (Map.Entry<String, List<String>> entry : request.getHeaders().entrySet()) {
            if (!entry.getKey().equalsIgnoreCase(header) || entry.getValue().size() <= 0) continue;
            return entry.getValue().get(0);
        }
        return null;
    }

    void updateClientStats(HttpRequest httpRequest, HttpChannel httpChannel) {
        if (this.clientStatsEnabled && httpChannel != null) {
            String opaqueId;
            String forwardedFor;
            HttpStats.ClientStats clientStats = this.httpChannelStats.get(HttpClientStatsTracker.getChannelKey(httpChannel));
            if (clientStats == null) {
                clientStats = this.addClientStats(httpChannel);
            }
            if (clientStats.agent == null) {
                String elasticProductOrigin = HttpClientStatsTracker.getFirstValueForHeader(httpRequest, "x-elastic-product-origin");
                if (elasticProductOrigin != null) {
                    clientStats.agent = elasticProductOrigin;
                } else {
                    String userAgent = HttpClientStatsTracker.getFirstValueForHeader(httpRequest, "User-Agent");
                    if (userAgent != null) {
                        clientStats.agent = userAgent;
                    }
                }
            }
            if (clientStats.localAddress == null) {
                clientStats.localAddress = httpChannel.getLocalAddress() == null ? null : NetworkAddress.format(httpChannel.getLocalAddress());
                String string = clientStats.remoteAddress = httpChannel.getRemoteAddress() == null ? null : NetworkAddress.format(httpChannel.getRemoteAddress());
            }
            if (clientStats.forwardedFor == null && (forwardedFor = HttpClientStatsTracker.getFirstValueForHeader(httpRequest, "x-forwarded-for")) != null) {
                clientStats.forwardedFor = forwardedFor;
            }
            if (clientStats.opaqueId == null && (opaqueId = HttpClientStatsTracker.getFirstValueForHeader(httpRequest, "x-opaque-id")) != null) {
                clientStats.opaqueId = opaqueId;
            }
            clientStats.lastRequestTimeMillis = this.threadPool.absoluteTimeInMillis();
            clientStats.lastUri = httpRequest.uri();
            clientStats.requestCount.increment();
            clientStats.requestSizeBytes.add(httpRequest.content().length());
        }
    }

    List<HttpStats.ClientStats> getClientStats() {
        this.pruneClientStats(false);
        return new ArrayList<HttpStats.ClientStats>(this.httpChannelStats.values());
    }

    private static int getChannelKey(HttpChannel channel) {
        return System.identityHashCode(channel);
    }
}

