/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.esapi.crypto;

import java.security.InvalidKeyException;
import java.security.InvalidParameterException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.Logger;
import org.owasp.esapi.crypto.CipherText;
import org.owasp.esapi.crypto.KeyDerivationFunction;
import org.owasp.esapi.errors.EncryptionException;

public class CryptoHelper {
    private static final Logger logger = ESAPI.getLogger("CryptoHelper");

    public static SecretKey generateSecretKey(String alg, int keySize) throws EncryptionException {
        assert (alg != null) : "Algorithm must not be null.";
        assert (!alg.equals("")) : "Algorithm must not be empty";
        assert (keySize > 0) : "Key size must be positive.";
        String[] cipherSpec = alg.split("/");
        String cipherAlg = cipherSpec[0];
        try {
            if (cipherAlg.toUpperCase().startsWith("PBEWITH")) {
                cipherAlg = "PBE";
            }
            KeyGenerator kgen = KeyGenerator.getInstance(cipherAlg);
            kgen.init(keySize);
            return kgen.generateKey();
        }
        catch (NoSuchAlgorithmException e) {
            throw new EncryptionException("Failed to generate random secret key", "Invalid algorithm. Failed to generate secret key for " + alg + " with size of " + keySize + " bits.", e);
        }
        catch (InvalidParameterException e) {
            throw new EncryptionException("Failed to generate random secret key - invalid key size specified.", "Invalid key size. Failed to generate secret key for " + alg + " with size of " + keySize + " bits.", e);
        }
    }

    public static SecretKey computeDerivedKey(SecretKey keyDerivationKey, int keySize, String purpose) throws NoSuchAlgorithmException, InvalidKeyException, EncryptionException {
        assert (keyDerivationKey != null) : "Key derivation key cannot be null.";
        assert (keySize >= 56) : "Key has size of " + keySize + ", which is less than minimum of 56-bits.";
        assert (keySize % 8 == 0) : "Key size (" + keySize + ") must be a even multiple of 8-bits.";
        assert (purpose != null);
        assert (purpose.equals("encryption") || purpose.equals("authenticity")) : "Purpose must be \"encryption\" or \"authenticity\".";
        KeyDerivationFunction kdf = new KeyDerivationFunction(KeyDerivationFunction.PRF_ALGORITHMS.HmacSHA1);
        return kdf.computeDerivedKey(keyDerivationKey, keySize, purpose);
    }

    public static boolean isCombinedCipherMode(String cipherMode) {
        assert (cipherMode != null) : "Cipher mode may not be null";
        assert (!cipherMode.equals("")) : "Cipher mode may not be empty string";
        List<String> combinedCipherModes = ESAPI.securityConfiguration().getCombinedCipherModes();
        return combinedCipherModes.contains(cipherMode);
    }

    public static boolean isAllowedCipherMode(String cipherMode) {
        if (CryptoHelper.isCombinedCipherMode(cipherMode)) {
            return true;
        }
        List<String> extraCipherModes = ESAPI.securityConfiguration().getAdditionalAllowedCipherModes();
        return extraCipherModes.contains(cipherMode);
    }

    public static boolean isMACRequired(CipherText ct) {
        boolean preferredCipherMode = CryptoHelper.isCombinedCipherMode(ct.getCipherMode());
        boolean wantsMAC = ESAPI.securityConfiguration().useMACforCipherText();
        return !preferredCipherMode && wantsMAC;
    }

    public static boolean isCipherTextMACvalid(SecretKey sk, CipherText ct) {
        if (CryptoHelper.isMACRequired(ct)) {
            try {
                SecretKey authKey = CryptoHelper.computeDerivedKey(sk, ct.getKeySize(), "authenticity");
                boolean validMAC = ct.validateMAC(authKey);
                return validMAC;
            }
            catch (Exception ex) {
                logger.warning(Logger.SECURITY_FAILURE, "Unable to validate MAC for ciphertext " + ct, ex);
                return false;
            }
        }
        return true;
    }

    public static void overwrite(byte[] bytes, byte x) {
        Arrays.fill(bytes, x);
    }

    public static void overwrite(byte[] bytes) {
        CryptoHelper.overwrite(bytes, (byte)42);
    }

    public static void copyByteArray(byte[] src, byte[] dest, int length) {
        System.arraycopy(src, 0, dest, 0, length);
    }

    public static void copyByteArray(byte[] src, byte[] dest) {
        CryptoHelper.copyByteArray(src, dest, src.length);
    }

    public static boolean arrayCompare(byte[] b1, byte[] b2) {
        if (b1 == b2) {
            return true;
        }
        if (b1 == null || b2 == null) {
            return b1 == b2;
        }
        if (b1.length != b2.length) {
            return false;
        }
        int result = 0;
        for (int i = 0; i < b1.length; ++i) {
            result |= b1[i] ^ b2[i];
        }
        return result == 0;
    }

    public static boolean isValidKDFVersion(int kdfVers, boolean restrictToCurrent, boolean throwIfError) throws IllegalArgumentException {
        boolean ret = true;
        if (kdfVers < 20110203 || kdfVers > 99991231) {
            ret = false;
        } else if (restrictToCurrent) {
            boolean bl = ret = kdfVers <= 20130830;
        }
        if (ret) {
            return ret;
        }
        logger.warning(Logger.SECURITY_FAILURE, "Possible data tampering. Encountered invalid KDF version #. " + (throwIfError ? "Throwing IllegalArgumentException" : ""));
        if (throwIfError) {
            throw new IllegalArgumentException("Version (" + kdfVers + ") invalid. " + "Must be date in format of YYYYMMDD between " + 20110203 + "and 99991231.");
        }
        return false;
    }

    private CryptoHelper() {
    }
}

