/*
 * Decompiled with CFR 0.152.
 */
package ucar.nc2.dt.fmrc;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.Formatter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.ma2.ArrayDouble;
import ucar.ma2.ArrayObject;
import ucar.ma2.DataType;
import ucar.ma2.InvalidRangeException;
import ucar.ma2.Range;
import ucar.ma2.Section;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.Group;
import ucar.nc2.NCdumpW;
import ucar.nc2.Variable;
import ucar.nc2.constants.AxisType;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.CoordinateAxis1DTime;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dataset.VariableDS;
import ucar.nc2.dt.GridCoordSystem;
import ucar.nc2.dt.GridDataset;
import ucar.nc2.dt.GridDatatype;
import ucar.nc2.dt.fmrc.ForecastModelRunCollection;
import ucar.nc2.units.DateFormatter;
import ucar.nc2.util.CancelTask;

public class FmrcImpl
implements ForecastModelRunCollection {
    private static Logger logger = LoggerFactory.getLogger(FmrcImpl.class);
    private static final String BEST = "best";
    private static final String RUN = "run";
    private static final String FORECAST = "forecast";
    private static final String OFFSET = "offset";
    private NetcdfDataset ncd_2dtime;
    private GridDataset gds;
    private Date baseDate;
    private String runtimeDimName;
    private List<Gridset> gridsets;
    private Map<String, Gridset> gridHash;
    private Set<String> coordSet;
    private List<Date> runtimes;
    private List<Date> forecasts;
    private List<Double> offsets;

    public FmrcImpl(String filename) throws IOException {
        this(NetcdfDataset.acquireDataset(filename, null));
    }

    public FmrcImpl(NetcdfDataset ncd) throws IOException {
        this.init(ncd);
    }

    @Override
    public boolean sync() throws IOException {
        boolean changed = this.ncd_2dtime.syncExtend();
        if (changed) {
            if (logger.isDebugEnabled()) {
                logger.debug("ncd_2dtime changed, reinit Fmrc " + this.ncd_2dtime.getLocation());
            }
            this.init(this.ncd_2dtime);
        }
        return changed;
    }

    @Override
    public GridDataset getGridDataset() {
        return this.gds;
    }

    @Override
    public void close() throws IOException {
        this.gds.close();
    }

    private void init(NetcdfDataset ncd) throws IOException {
        this.ncd_2dtime = ncd;
        this.gridHash = new HashMap<String, Gridset>();
        this.coordSet = new HashSet<String>();
        this.runtimes = null;
        this.gds = new ucar.nc2.dt.grid.GridDataset(ncd);
        List<GridDatatype> grids = this.gds.getGrids();
        if (grids.size() == 0) {
            throw new IllegalArgumentException("no grids");
        }
        HashMap<CoordinateAxis, Gridset> timeAxisHash = new HashMap<CoordinateAxis, Gridset>();
        for (GridDatatype grid : grids) {
            GridCoordSystem gcs = grid.getCoordinateSystem();
            CoordinateAxis timeAxis = gcs.getTimeAxis();
            if (timeAxis != null) {
                Gridset gset = (Gridset)timeAxisHash.get(timeAxis);
                if (gset == null) {
                    gset = new Gridset(timeAxis, gcs);
                    timeAxisHash.put(timeAxis, gset);
                    this.coordSet.add(timeAxis.getFullName());
                }
                gset.gridList.add(grid);
                this.gridHash.put(grid.getFullName(), gset);
            }
            if (this.runtimes != null || gcs.getRunTimeAxis() == null) continue;
            CoordinateAxis1DTime runtimeCoord = gcs.getRunTimeAxis();
            Date[] runDates = runtimeCoord.getTimeDates();
            this.baseDate = runDates[0];
            this.runtimes = Arrays.asList(runDates);
            this.runtimeDimName = runtimeCoord.getDimension(0).getShortName();
            this.coordSet.add(runtimeCoord.getFullName());
        }
        if (this.runtimes == null) {
            throw new IllegalArgumentException("no runtime dimension");
        }
        HashSet<Date> forecastSet = new HashSet<Date>();
        HashSet<Double> offsetSet = new HashSet<Double>();
        this.gridsets = new ArrayList(timeAxisHash.values());
        for (Gridset gridset : this.gridsets) {
            for (int run = 0; run < this.runtimes.size(); ++run) {
                Date[] forecastDates;
                Date runDate = this.runtimes.get(run);
                CoordinateAxis1DTime timeCoordRun = gridset.gcs.getTimeAxisForRun(run);
                for (Date forecastDate : forecastDates = timeCoordRun.getTimeDates()) {
                    forecastSet.add(forecastDate);
                    double hourOffset = this.getOffsetHour(runDate, forecastDate);
                    offsetSet.add(hourOffset);
                }
            }
        }
        this.forecasts = Arrays.asList(forecastSet.toArray(new Date[forecastSet.size()]));
        Collections.sort(this.forecasts);
        this.offsets = Arrays.asList(offsetSet.toArray(new Double[offsetSet.size()]));
        Collections.sort(this.offsets);
        for (Gridset gridset : this.gridsets) {
            gridset.generateInventory();
        }
    }

    private double getOffsetHour(Date run, Date forecast) {
        double diff = forecast.getTime() - run.getTime();
        return diff / 1000.0 / 60.0 / 60.0;
    }

    @Override
    public List<Date> getRunDates() {
        return this.runtimes;
    }

    @Override
    public NetcdfDataset getRunTimeDataset(Date wantRuntime) throws IOException {
        if (wantRuntime == null) {
            return null;
        }
        if (!this.runtimes.contains(wantRuntime)) {
            return null;
        }
        DateFormatter df = new DateFormatter();
        String runTimeString = df.toDateTimeStringISO(wantRuntime);
        NetcdfDataset ncd = this.createDataset(new RuntimeInvGetter(wantRuntime), RUN, runTimeString);
        ncd.addAttribute(null, new Attribute("_CoordinateModelRunDate", runTimeString));
        ncd.finish();
        return ncd;
    }

    @Override
    public List<Date> getForecastDates() {
        return this.forecasts;
    }

    @Override
    public NetcdfDataset getForecastTimeDataset(Date forecastTime) throws IOException {
        if (forecastTime == null) {
            return null;
        }
        if (!this.forecasts.contains(forecastTime)) {
            return null;
        }
        DateFormatter df = new DateFormatter();
        String name = df.toDateTimeStringISO(forecastTime);
        return this.createDataset(new ForecastInvGetter(forecastTime), FORECAST, name);
    }

    @Override
    public List<Double> getForecastOffsets() {
        return this.offsets;
    }

    @Override
    public NetcdfDataset getForecastOffsetDataset(double hours) throws IOException {
        if (!this.offsets.contains(new Double(hours))) {
            return null;
        }
        return this.createDataset(new OffsetInvGetter(hours), OFFSET, Double.toString(hours));
    }

    @Override
    public NetcdfDataset getBestTimeSeries() throws IOException {
        return this.createDataset(new InventoryGetter(){

            @Override
            public List<Inventory> get(Gridset gridset) {
                return gridset.bestList;
            }
        }, BEST, null);
    }

    @Override
    public NetcdfDataset getFmrcDataset() {
        return this.ncd_2dtime;
    }

    private String makeLocation(String type, String name) {
        if (name != null) {
            return this.ncd_2dtime.getLocation() + "/" + type + "-" + name + ".ncd";
        }
        return this.ncd_2dtime.getLocation() + "/" + type + ".ncd";
    }

    private NetcdfDataset createDataset(InventoryGetter invGetter, String type, String name) throws IOException {
        NetcdfDataset newds = new NetcdfDataset();
        newds.setLocation(this.makeLocation(type, name));
        Group src = this.ncd_2dtime.getRootGroup();
        Group target = newds.getRootGroup();
        for (Attribute a : src.getAttributes()) {
            target.addAttribute(a);
        }
        String oldHistory = this.ncd_2dtime.findAttValueIgnoreCase(null, "history", null);
        String newHistory = "Synthetic dataset from TDS fmrc (" + type + ") aggregation, original data from " + this.ncd_2dtime.getLocation();
        String history = oldHistory != null ? oldHistory + "; " + newHistory : newHistory;
        target.addAttribute(new Attribute("history", history));
        DateFormatter df = new DateFormatter();
        target.addAttribute(new Attribute("_CoordinateModelBaseDate", df.toDateTimeStringISO(this.baseDate)));
        for (Dimension d : src.getDimensions()) {
            target.addDimension(new Dimension(d.getShortName(), d));
        }
        for (Gridset gridset : this.gridsets) {
            List<Inventory> invList = invGetter.get(gridset);
            if (invList == null) continue;
            this.addTime3Coordinates(newds, gridset, invList, type);
            for (GridDatatype grid : gridset.gridList) {
                Variable orgVar = this.ncd_2dtime.findVariable(grid.getVariable().getFullNameEscaped());
                VariableDS v = new VariableDS(target, orgVar, false);
                v.clearCoordinateSystems();
                v.setDimensions(gridset.makeDimensions(v.getDimensions()));
                v.remove(v.findAttribute("_CoordinateAxes"));
                v.remove(v.findAttribute("coordinates"));
                String coords = this.makeCoordinatesAttribute(grid.getCoordinateSystem(), gridset.timeDimName);
                v.addAttribute(new Attribute("coordinates", coords));
                target.addVariable(v);
            }
        }
        for (Variable v : src.getVariables()) {
            if (null != this.gridHash.get(v.getFullName()) || this.coordSet.contains(v.getFullName())) continue;
            VariableDS vds = new VariableDS(newds.getRootGroup(), v, false);
            vds.clearCoordinateSystems();
            vds.remove(vds.findAttribute("coordinates"));
            target.addVariable(vds);
        }
        newds.finish();
        newds.enhance(EnumSet.of(NetcdfDataset.Enhance.CoordSystems));
        return newds;
    }

    private String makeCoordinatesAttribute(GridCoordSystem gcs, String timeDimName) {
        Formatter sb = new Formatter();
        if (gcs.getXHorizAxis() != null) {
            sb.format("%s ", gcs.getXHorizAxis().getFullName());
        }
        if (gcs.getYHorizAxis() != null) {
            sb.format("%s ", gcs.getYHorizAxis().getFullName());
        }
        if (gcs.getVerticalAxis() != null) {
            sb.format("%s ", gcs.getVerticalAxis().getFullName());
        }
        sb.format("%s ", timeDimName);
        return sb.toString();
    }

    private void addTime3Coordinates(NetcdfDataset newds, Gridset gridset, List<Inventory> invList, String type) {
        DateFormatter formatter = new DateFormatter();
        boolean useRun = type.equals(FORECAST);
        int n = invList.size();
        String dimName = gridset.timeDimName;
        Group g = newds.getRootGroup();
        g.remove(g.findDimension(dimName));
        g.addDimension(new Dimension(dimName, n));
        ArrayDouble.D1 offsetData = new ArrayDouble.D1(n);
        for (int i = 0; i < n; ++i) {
            Inventory inv = invList.get(i);
            double offsetHour = this.getOffsetHour(this.baseDate, useRun ? inv.runTime : inv.forecastTime);
            offsetData.set(i, offsetHour);
        }
        String typeName = useRun ? RUN : FORECAST;
        String desc = typeName + " time coordinate";
        VariableDS timeCoordinate = new VariableDS(newds, g, null, dimName, DataType.DOUBLE, dimName, "hours since " + formatter.toDateTimeStringISO(this.baseDate), desc);
        timeCoordinate.setCachedData(offsetData, true);
        timeCoordinate.addAttribute(new Attribute("long_name", desc));
        timeCoordinate.addAttribute(new Attribute("standard_name", useRun ? "forecast_reference_time" : "time"));
        timeCoordinate.addAttribute(new Attribute("_CoordinateAxisType", AxisType.Time.toString()));
        newds.addVariable(g, timeCoordinate);
        ArrayObject.D1 runData = new ArrayObject.D1(String.class, n);
        for (int i = 0; i < n; ++i) {
            Inventory inv = invList.get(i);
            runData.set(i, formatter.toDateTimeStringISO(inv.runTime));
        }
        desc = "model run dates for coordinate = " + dimName;
        VariableDS runtimeCoordinate = new VariableDS(newds, newds.getRootGroup(), null, dimName + "_run", DataType.STRING, dimName, null, desc);
        runtimeCoordinate.setCachedData(runData, true);
        runtimeCoordinate.addAttribute(new Attribute("long_name", desc));
        runtimeCoordinate.addAttribute(new Attribute("standard_name", "forecast_reference_time"));
        runtimeCoordinate.addAttribute(new Attribute("_CoordinateAxisType", AxisType.RunTime.toString()));
        newds.addVariable(newds.getRootGroup(), runtimeCoordinate);
        offsetData = new ArrayDouble.D1(n);
        for (int i = 0; i < n; ++i) {
            Inventory inv = invList.get(i);
            offsetData.set(i, inv.hourOffset);
        }
        desc = "hour offset from start of run for coordinate = " + dimName;
        VariableDS offsetCoordinate = new VariableDS(newds, newds.getRootGroup(), null, dimName + "_offset", DataType.DOUBLE, dimName, null, desc);
        offsetCoordinate.setCachedData(offsetData, true);
        offsetCoordinate.addAttribute(new Attribute("long_name", desc));
        offsetCoordinate.addAttribute(new Attribute("units", "hour"));
        offsetCoordinate.addAttribute(new Attribute("standard_name", "forecast_period"));
        newds.addVariable(newds.getRootGroup(), offsetCoordinate);
    }

    public void dump(Formatter f) throws IOException {
        for (Gridset gridset : this.gridsets) {
            f.format("===========================%n", new Object[0]);
            gridset.dump(f);
        }
    }

    static void test(String location, String timeVarName) throws IOException {
        FmrcImpl fmrc = new FmrcImpl(location);
        System.out.println("Fmrc for dataset= " + location);
        NetcdfDataset fmrcd = fmrc.getFmrcDataset();
        Variable time = fmrcd.findVariable(timeVarName);
        Array data = time.read();
        NCdumpW.printArray(data, "2D time", new PrintWriter(System.out), null);
        fmrc.dump(new Formatter(System.out));
    }

    public static void main(String[] args) throws Exception {
        FmrcImpl.test("D:/test/signell/test.ncml", "ocean_time");
    }

    private class Subsetter {
        List<Inventory> invList;
        Variable mainv;

        Subsetter(List<Inventory> invList, Variable mainv) {
            this.invList = invList;
            this.mainv = mainv;
        }

        public Array reallyRead(CancelTask cancelTask) throws IOException {
            Variable orgVar = FmrcImpl.this.ncd_2dtime.findVariable(this.mainv.getFullNameEscaped());
            int[] orgVarShape = orgVar.getShape();
            int rank = orgVar.getRank() - 1;
            int[] varShape = new int[rank];
            varShape[0] = this.invList.size();
            System.arraycopy(orgVarShape, 2, varShape, 1, rank - 1);
            Array allData = Array.factory(this.mainv.getDataType(), varShape);
            int destPos = 0;
            Section section = new Section(orgVar.getRanges());
            for (Inventory inv : this.invList) {
                Array varData;
                try {
                    section.setRange(0, new Range(inv.run, inv.run));
                    section.setRange(1, new Range(inv.time, inv.time));
                    varData = orgVar.read(section);
                }
                catch (InvalidRangeException e) {
                    logger.error("read failed", (Throwable)e);
                    throw new IllegalStateException(e.getMessage());
                }
                Array.arraycopy(varData, 0, allData, destPos, (int)varData.getSize());
                destPos = (int)((long)destPos + varData.getSize());
                if (cancelTask == null || !cancelTask.isCancel()) continue;
                return null;
            }
            return allData;
        }

        public Array reallyRead(Section section, CancelTask cancelTask) throws IOException, InvalidRangeException {
            long size = section.computeSize();
            if (size == this.mainv.getSize()) {
                return this.reallyRead(cancelTask);
            }
            Variable orgVar = FmrcImpl.this.ncd_2dtime.findVariable(this.mainv.getFullNameEscaped());
            Array sectionData = Array.factory(this.mainv.getDataType(), section.getShape());
            int destPos = 0;
            Range timeRange = section.getRange(0);
            ArrayList<Range> allSection = new ArrayList<Range>(section.getRanges());
            allSection.add(0, null);
            Range.Iterator iter = timeRange.getIterator();
            while (iter.hasNext()) {
                Array varData;
                int index = iter.next();
                Inventory inv = this.invList.get(index);
                try {
                    allSection.set(0, new Range(inv.run, inv.run));
                    allSection.set(1, new Range(inv.time, inv.time));
                    varData = orgVar.read(allSection);
                }
                catch (InvalidRangeException e) {
                    logger.error("readSection failed", (Throwable)e);
                    throw new IllegalStateException("read failed", e);
                }
                Array.arraycopy(varData, 0, sectionData, destPos, (int)varData.getSize());
                destPos = (int)((long)destPos + varData.getSize());
                if (cancelTask == null || !cancelTask.isCancel()) continue;
                return null;
            }
            return sectionData;
        }
    }

    private class OffsetInvGetter
    implements InventoryGetter {
        Double hours;

        OffsetInvGetter(double hours) {
            this.hours = hours;
        }

        @Override
        public List<Inventory> get(Gridset gridset) {
            return gridset.offsetMap.get(this.hours);
        }
    }

    private class ForecastInvGetter
    implements InventoryGetter {
        Date forecastTime;

        ForecastInvGetter(Date forecastTime) {
            this.forecastTime = forecastTime;
        }

        @Override
        public List<Inventory> get(Gridset gridset) {
            return gridset.timeMap.get(this.forecastTime);
        }
    }

    private class RuntimeInvGetter
    implements InventoryGetter {
        Date wantRuntime;

        RuntimeInvGetter(Date wantRuntime) {
            this.wantRuntime = wantRuntime;
        }

        @Override
        public List<Inventory> get(Gridset gridset) {
            return gridset.runMap.get(this.wantRuntime);
        }
    }

    private static interface InventoryGetter {
        public List<Inventory> get(Gridset var1);
    }

    private class Inventory
    implements Comparable {
        Date forecastTime;
        Date runTime;
        double hourOffset;
        int run;
        int time;

        Inventory(Date runTime, Date forecastTime, double hourOffset, int run, int time) {
            this.runTime = runTime;
            this.hourOffset = hourOffset;
            this.forecastTime = forecastTime;
            this.run = run;
            this.time = time;
        }

        public int compareTo(Object o) {
            Inventory other = (Inventory)o;
            return this.forecastTime.compareTo(other.forecastTime);
        }
    }

    private class Gridset {
        List<GridDatatype> gridList = new ArrayList<GridDatatype>();
        GridCoordSystem gcs;
        CoordinateAxis timeAxis;
        String timeDimName;
        HashMap<Date, List<Inventory>> runMap = new HashMap();
        HashMap<Date, List<Inventory>> timeMap = new HashMap();
        HashMap<Double, List<Inventory>> offsetMap = new HashMap();
        List<Inventory> bestList = new ArrayList<Inventory>();

        Gridset(CoordinateAxis timeAxis, GridCoordSystem gcs) {
            this.gcs = gcs;
            this.timeAxis = timeAxis;
            this.timeDimName = timeAxis.getDimension(1).getShortName();
        }

        String makeDimensions(List<Dimension> dims) {
            StringBuilder sbuff = new StringBuilder();
            sbuff.append(this.timeDimName);
            for (Dimension d : dims) {
                if (d.getShortName().equals(FmrcImpl.this.runtimeDimName) || d.getShortName().equals(this.timeDimName)) continue;
                sbuff.append(" ").append(d.getShortName());
            }
            return sbuff.toString();
        }

        void generateInventory() {
            HashMap<Date, Inventory> bestMap = new HashMap<Date, Inventory>();
            int nruns = FmrcImpl.this.runtimes.size();
            for (int run = 0; run < nruns; ++run) {
                Date runDate = (Date)FmrcImpl.this.runtimes.get(run);
                ArrayList<Inventory> runList = new ArrayList<Inventory>();
                this.runMap.put(runDate, runList);
                CoordinateAxis1DTime timeCoordRun = this.gcs.getTimeAxisForRun(run);
                Date[] forecastDates = timeCoordRun.getTimeDates();
                for (int time = 0; time < forecastDates.length; ++time) {
                    Date forecastDate = forecastDates[time];
                    double hourOffset = FmrcImpl.this.getOffsetHour(runDate, forecastDate);
                    Inventory inv = new Inventory(runDate, forecastDate, hourOffset, run, time);
                    runList.add(inv);
                    bestMap.put(forecastDate, inv);
                    List<Inventory> offsetList = this.offsetMap.get(hourOffset);
                    if (offsetList == null) {
                        offsetList = new ArrayList<Inventory>();
                        this.offsetMap.put(hourOffset, offsetList);
                    }
                    offsetList.add(inv);
                    List<Inventory> timeList = this.timeMap.get(forecastDate);
                    if (timeList == null) {
                        timeList = new ArrayList<Inventory>();
                        this.timeMap.put(forecastDate, timeList);
                    }
                    timeList.add(inv);
                }
            }
            this.bestList = new ArrayList(bestMap.values());
            Collections.sort(this.bestList);
        }

        void dump(Formatter f) throws IOException {
            List<Inventory> list;
            DateFormatter df = new DateFormatter();
            f.format("Gridset timeDimName= %s%n grids= %n", this.timeDimName);
            for (GridDatatype grid : this.gridList) {
                f.format("  %s%n", grid.getFullName());
            }
            f.format("%nRun Dates= %s%n", FmrcImpl.this.runtimes.size());
            for (Date date : FmrcImpl.this.runtimes) {
                f.format(" %s (", df.toDateTimeString(date));
                list = this.runMap.get(date);
                if (list == null) {
                    f.format(" none", new Object[0]);
                } else {
                    for (Inventory inv : list) {
                        f.format(" %s", inv.hourOffset);
                    }
                }
                f.format(") %n", new Object[0]);
            }
            f.format("%nForecast Dates= %d %n", FmrcImpl.this.forecasts.size());
            for (Date date : FmrcImpl.this.forecasts) {
                f.format(" %s(", df.toDateTimeString(date));
                list = this.timeMap.get(date);
                if (list == null) {
                    f.format(" none", new Object[0]);
                } else {
                    for (Inventory inv : list) {
                        f.format(" %d/%f", inv.run, inv.hourOffset);
                    }
                }
                f.format(")%n", new Object[0]);
            }
            f.format("\nForecast Hours= %d%n", FmrcImpl.this.offsets.size());
            for (Double hour : FmrcImpl.this.offsets) {
                List<Inventory> offsetList = this.offsetMap.get(hour);
                f.format(" %s: (", hour);
                if (offsetList == null) {
                    f.format(" none", new Object[0]);
                } else {
                    for (int j = 0; j < offsetList.size(); ++j) {
                        Inventory inv;
                        inv = offsetList.get(j);
                        if (j > 0) {
                            System.out.print(", ");
                        }
                        f.format("%d/%s", inv.run, df.toDateTimeStringISO(inv.runTime));
                    }
                }
                f.format(")%n", new Object[0]);
            }
            f.format("\nBest Forecast = %d%n", this.bestList.size());
            for (Inventory inv : this.bestList) {
                f.format(" %s (run=%s) offset=%f%n", df.toDateTimeStringISO(inv.forecastTime), df.toDateTimeStringISO(inv.runTime), inv.hourOffset);
            }
        }
    }
}

